"use server";

import { requireSession } from "@/lib/session";
import { prisma } from "@/lib/prisma";
import { hashApiKey, randomToken } from "@/lib/crypto";
import { redirect } from "next/navigation";
import { z } from "zod";

const CreateKeySchema = z.object({
  name: z.string().min(1).max(40),
});

const CreateWebhookSchema = z.object({
  url: z.string().url().max(2000),
  events: z.string().max(200).optional().default(""),
});

const CreateLocationSchema = z.object({
  name: z.string().min(1).max(80),
  city: z.string().max(80).optional(),
  address: z.string().max(200).optional(),
});

const CreateChargeSchema = z.object({
  locationId: z.string().min(1),
  amount: z.coerce.number().positive().max(100000),
  reference: z.string().max(64).optional(),
  description: z.string().max(140).optional(),
});

const ShiftSchema = z.object({
  locationId: z.string().min(1),
  note: z.string().max(140).optional(),
  shiftId: z.string().optional(),
});

export async function createApiKeyAction(_: any, formData: FormData) {
  const session = await requireSession();

  const parsed = CreateKeySchema.safeParse({ name: formData.get("name") });
  if (!parsed.success) return { ok: false, error: "INVALID_INPUT" };

  const apiKeyPlain = `lefa_${randomToken(16)}`;
  const keyHash = await hashApiKey(apiKeyPlain);

  await prisma.apiKey.create({
    data: {
      userId: session.uid,
      name: parsed.data.name,
      keyHash,
    },
  });

  return { ok: true, apiKey: apiKeyPlain };
}

export async function revokeApiKeyAction(_: any, formData: FormData) {
  const session = await requireSession();
  const id = String(formData.get("id") ?? "");
  if (!id) return { ok: false, error: "INVALID_INPUT" };

  await prisma.apiKey.updateMany({
    where: { id, userId: session.uid, revokedAt: null },
    data: { revokedAt: new Date() },
  });

  return { ok: true };
}

export async function createWebhookAction(_: any, formData: FormData) {
  const session = await requireSession();
  const parsed = CreateWebhookSchema.safeParse({
    url: formData.get("url"),
    events: formData.get("events"),
  });
  if (!parsed.success) return { ok: false, error: "INVALID_INPUT" };

  const secret = `whsec_${randomToken(18)}`;

  await prisma.webhookEndpoint.create({
    data: {
      userId: session.uid,
      url: parsed.data.url,
      events: parsed.data.events,
      secret,
    },
  });

  return { ok: true, secret };
}

export async function disableWebhookAction(_: any, formData: FormData) {
  const session = await requireSession();
  const id = String(formData.get("id") ?? "");
  if (!id) return { ok: false, error: "INVALID_INPUT" };

  await prisma.webhookEndpoint.updateMany({
    where: { id, userId: session.uid, disabledAt: null },
    data: { disabledAt: new Date() },
  });

  return { ok: true };
}

export async function createLocationAction(_: any, formData: FormData) {
  const session = await requireSession();
  const parsed = CreateLocationSchema.safeParse({
    name: formData.get("name"),
    city: formData.get("city"),
    address: formData.get("address"),
  });
  if (!parsed.success) return { ok: false, error: "INVALID_INPUT" };

  const merchant = await prisma.merchantProfile.findUnique({ where: { userId: session.uid } });
  if (!merchant) return { ok: false, error: "NOT_MERCHANT" };

  const loc = await prisma.merchantLocation.create({
    data: {
      merchantId: merchant.id,
      name: parsed.data.name,
      city: parsed.data.city,
      address: parsed.data.address,
    },
  });
  return { ok: true, locationId: loc.id };
}

export async function openShiftAction(_: any, formData: FormData) {
  const session = await requireSession();
  const parsed = ShiftSchema.safeParse({
    locationId: formData.get("locationId"),
    note: formData.get("note"),
  });
  if (!parsed.success) return { ok: false, error: "INVALID_INPUT" };
  const merchant = await prisma.merchantProfile.findUnique({ where: { userId: session.uid } });
  if (!merchant) return { ok: false, error: "NOT_MERCHANT" };

  // Close any open shift for this cashier/location
  await prisma.cashierShift.updateMany({
    where: { merchantId: merchant.id, locationId: parsed.data.locationId, cashierUserId: session.uid, closedAt: null },
    data: { closedAt: new Date() },
  });

  const shift = await prisma.cashierShift.create({
    data: {
      merchantId: merchant.id,
      locationId: parsed.data.locationId,
      cashierUserId: session.uid,
      note: parsed.data.note,
    },
  });
  return { ok: true, shiftId: shift.id };
}

export async function closeShiftAction(_: any, formData: FormData) {
  const session = await requireSession();
  const parsed = ShiftSchema.safeParse({
    shiftId: formData.get("shiftId"),
  });
  if (!parsed.success || !parsed.data.shiftId) return { ok: false, error: "INVALID_INPUT" };
  const merchant = await prisma.merchantProfile.findUnique({ where: { userId: session.uid } });
  if (!merchant) return { ok: false, error: "NOT_MERCHANT" };

  await prisma.cashierShift.updateMany({
    where: { id: parsed.data.shiftId, merchantId: merchant.id, cashierUserId: session.uid, closedAt: null },
    data: { closedAt: new Date() },
  });
  return { ok: true };
}

export async function createInStoreChargeAction(_: any, formData: FormData) {
  const session = await requireSession();
  const parsed = CreateChargeSchema.safeParse({
    locationId: formData.get("locationId"),
    amount: formData.get("amount"),
    reference: formData.get("reference"),
    description: formData.get("description"),
  });
  if (!parsed.success) return { ok: false, error: "INVALID_INPUT" };
  const merchant = await prisma.merchantProfile.findUnique({ where: { userId: session.uid } });
  if (!merchant) return { ok: false, error: "NOT_MERCHANT" };

  const amountMinor = BigInt(Math.round(parsed.data.amount * 100));
  const token = `pos_${randomToken(12)}`;

  const openShift = await prisma.cashierShift.findFirst({
    where: { merchantId: merchant.id, locationId: parsed.data.locationId, cashierUserId: session.uid, closedAt: null },
    orderBy: { openedAt: "desc" },
  });

  const charge = await prisma.inStoreCharge.create({
    data: {
      merchantId: merchant.id,
      locationId: parsed.data.locationId,
      shiftId: openShift?.id,
      amount: amountMinor,
      currency: "BWP",
      reference: parsed.data.reference,
      description: parsed.data.description,
      token,
    },
  });

  redirect(`/merchant/pos/qr/${charge.token}`);
}
