import { z } from "zod";
import { prisma } from "@/lib/prisma";
import { getGuestSession } from "@/lib/guest";
import { settleExternalPayment } from "@/lib/ledger";
import { dispatchUserWebhooks } from "@/lib/webhooks";

const BodySchema = z.object({
  checkoutToken: z.string().min(1),
});

export async function POST(req: Request) {
  const guest = getGuestSession();
  if (!guest) return Response.json({ error: "GUEST_UNAUTHENTICATED" }, { status: 401 });

  const json = await req.json().catch(() => null);
  const parsed = BodySchema.safeParse(json);
  if (!parsed.success) return Response.json({ error: "INVALID_BODY" }, { status: 400 });

  const pi = await prisma.paymentIntent.findUnique({
    where: { checkoutToken: parsed.data.checkoutToken },
    include: { merchant: { include: { user: true } } },
  });
  if (!pi) return Response.json({ error: "NOT_FOUND" }, { status: 404 });
  if (pi.status === "SUCCEEDED") return Response.json({ ok: true, status: pi.status });

  const guestRow = await prisma.guestPayer.findUnique({ where: { id: guest.guestId } });
  if (!guestRow) return Response.json({ error: "GUEST_NOT_FOUND" }, { status: 401 });

  const ref = `PI-${pi.reference}`;
  const res = await settleExternalPayment({
    toUserId: pi.merchant.userId,
    amountMinor: pi.amount,
    description: pi.description ?? `Guest payment ${pi.reference}`,
    reference: ref,
  });

  const updated = await prisma.paymentIntent.update({
    where: { id: pi.id },
    data: {
      status: "SUCCEEDED",
      paidByGuestId: guestRow.id,
    },
  });

  await dispatchUserWebhooks(pi.merchant.userId, "payment_intent.succeeded", {
    checkoutToken: pi.checkoutToken,
    amount: pi.amount.toString(),
    currency: pi.currency,
    reference: pi.reference,
    txId: res.transaction.id,
    guest: { id: guestRow.id, phone: guestRow.phone },
  });

  return Response.json({ ok: true, status: updated.status, txId: res.transaction.id });
}
