import { z } from "zod";
import { prisma } from "@/lib/prisma";
import { setGuestSession } from "@/lib/guest";

const BodySchema = z.object({
  challengeId: z.string().min(1),
  code: z.string().min(4).max(10),
  phone: z.string().min(7).max(20),
});

export async function POST(req: Request) {
  const json = await req.json().catch(() => null);
  const parsed = BodySchema.safeParse(json);
  if (!parsed.success) return Response.json({ error: "INVALID_BODY" }, { status: 400 });

  const { challengeId, code, phone } = parsed.data;

  const ch = await prisma.otpChallenge.findUnique({ where: { id: challengeId } });
  if (!ch) return Response.json({ error: "CHALLENGE_NOT_FOUND" }, { status: 404 });
  if (ch.phone !== phone.trim()) return Response.json({ error: "MISMATCH" }, { status: 400 });
  if (ch.verifiedAt) return Response.json({ error: "ALREADY_VERIFIED" }, { status: 400 });
  if (ch.expiresAt.getTime() < Date.now()) return Response.json({ error: "EXPIRED" }, { status: 400 });
  if (ch.code !== code.trim()) return Response.json({ error: "INVALID_CODE" }, { status: 401 });

  await prisma.otpChallenge.update({ where: { id: ch.id }, data: { verifiedAt: new Date() } });

  const guest = await prisma.guestPayer.create({
    data: { phone: phone.trim() },
  });

  setGuestSession({ guestId: guest.id, phone: guest.phone ?? undefined });

  return Response.json({ ok: true, guestId: guest.id });
}
