import { z } from "zod";
import { prisma } from "@/lib/prisma";
import { requireSession } from "@/lib/session";

const BodySchema = z.object({
  challengeId: z.string().min(1),
  otp: z.string().min(4).max(10),
  phone: z.string().min(7).max(20),
});

export async function POST(req: Request) {
  const session = await requireSession();
  const json = await req.json().catch(() => null);
  const parsed = BodySchema.safeParse(json);
  if (!parsed.success) return Response.json({ error: "INVALID_BODY" }, { status: 400 });

  const { challengeId, otp, phone } = parsed.data;
  const ch = await prisma.otpChallenge.findUnique({ where: { id: challengeId } });
  if (!ch || ch.purpose !== "KYC_PHONE") return Response.json({ error: "NOT_FOUND" }, { status: 404 });
  if (ch.phone !== phone.trim()) return Response.json({ error: "PHONE_MISMATCH" }, { status: 400 });
  if (ch.verifiedAt) return Response.json({ error: "ALREADY_VERIFIED" }, { status: 400 });
  if (ch.expiresAt.getTime() < Date.now()) return Response.json({ error: "EXPIRED" }, { status: 400 });
  if (ch.code !== otp.trim()) return Response.json({ error: "INVALID_OTP" }, { status: 400 });

  await prisma.otpChallenge.update({ where: { id: ch.id }, data: { verifiedAt: new Date() } });
  await prisma.kycProfile.upsert({
    where: { userId: session.uid },
    update: { phoneVerifiedAt: new Date(), status: "APPROVED" },
    create: { userId: session.uid, phoneVerifiedAt: new Date(), status: "APPROVED" },
  });
  await prisma.user.update({ where: { id: session.uid }, data: { kycLevel: "BASIC", phone: phone.trim() } });

  return Response.json({ ok: true });
}
