import { z } from "zod";
import { prisma } from "@/lib/prisma";
import { fromMinor } from "@/lib/money";
import { transferLefaToLefa } from "@/lib/ledger";

const BodySchema = z.object({
  sessionId: z.string().min(1),
  input: z.string().max(200).optional().default(""),
});

function mainMenu() {
  return [
    "LEFA *555#",
    "1. Balance",
    "2. Send money",
    "3. Pay merchant",
    "4. Recent transactions",
    "5. KYC status",
    "0. Exit",
  ].join("\n");
}

export async function POST(req: Request) {
  const json = await req.json().catch(() => null);
  const parsed = BodySchema.safeParse(json);
  if (!parsed.success) return Response.json({ error: "INVALID_BODY" }, { status: 400 });

  const s = await prisma.ussdSession.findUnique({ where: { id: parsed.data.sessionId } });
  if (!s) return Response.json({ error: "SESSION_NOT_FOUND" }, { status: 404 });

  const input = parsed.data.input.trim();
  const state: any = s.state ?? {};

  // Resolve user by phone (lite)
  const user = await prisma.user.findFirst({ where: { phone: s.phone }, include: { wallet: true, kycProfile: true } });

  // MAIN
  if (s.lastMenu === "MAIN") {
    if (input === "0") return Response.json({ ok: true, menu: "Thank you for using LEFA." });
    if (input === "1") {
      const bal = user?.wallet?.balance ?? 0n;
      return Response.json({ ok: true, menu: `Balance: P ${fromMinor(bal)}\n\n0. Back` });
    }
    if (input === "2") {
      await prisma.ussdSession.update({ where: { id: s.id }, data: { lastMenu: "SEND_EMAIL", state: { step: "SEND_EMAIL" } } });
      return Response.json({ ok: true, menu: "Send money\nEnter recipient email:" });
    }
    if (input === "3") {
      await prisma.ussdSession.update({ where: { id: s.id }, data: { lastMenu: "PAY_MERCHANT", state: { step: "PAY_MERCHANT" } } });
      return Response.json({ ok: true, menu: "Pay merchant\nEnter merchant email:" });
    }
    if (input === "4") {
      if (!user) return Response.json({ ok: true, menu: "No wallet linked to this phone.\n\n0. Back" });
      const txs = await prisma.transaction.findMany({
        where: { OR: [{ fromUserId: user.id }, { toUserId: user.id }] },
        orderBy: { createdAt: "desc" },
        take: 5,
      });
      const lines = txs.map((t) => `${t.type} P${fromMinor(t.amount)} ${t.status}`);
      return Response.json({ ok: true, menu: ["Recent:", ...lines, "", "0. Back"].join("\n") });
    }
    if (input === "5") {
      if (!user) return Response.json({ ok: true, menu: "No account linked to this phone.\n\n0. Back" });
      return Response.json({ ok: true, menu: `KYC level: ${user.kycLevel}\nPhone verified: ${user.kycProfile?.phoneVerifiedAt ? "YES" : "NO"}\n\n0. Back` });
    }
    return Response.json({ ok: true, menu: mainMenu() });
  }

  // SEND_EMAIL
  if (s.lastMenu === "SEND_EMAIL") {
    if (input === "0") {
      await prisma.ussdSession.update({ where: { id: s.id }, data: { lastMenu: "MAIN", state: { step: "MAIN" } } });
      return Response.json({ ok: true, menu: mainMenu() });
    }
    await prisma.ussdSession.update({
      where: { id: s.id },
      data: { lastMenu: "SEND_AMOUNT", state: { step: "SEND_AMOUNT", toEmail: input } },
    });
    return Response.json({ ok: true, menu: `To: ${input}\nEnter amount (BWP):` });
  }

  // SEND_AMOUNT
  if (s.lastMenu === "SEND_AMOUNT") {
    if (input === "0") {
      await prisma.ussdSession.update({ where: { id: s.id }, data: { lastMenu: "MAIN", state: { step: "MAIN" } } });
      return Response.json({ ok: true, menu: mainMenu() });
    }
    if (!user) return Response.json({ ok: true, menu: "No account linked to this phone." });
    const amount = Number(input);
    if (!Number.isFinite(amount) || amount <= 0) return Response.json({ ok: true, menu: "Invalid amount. Enter amount (BWP):" });
    const amountMinor = BigInt(Math.round(amount * 100));
    const toEmail = String(state.toEmail ?? "");
    try {
      await transferLefaToLefa({ fromUserId: user.id, toEmail, amountMinor, description: "USSD send" });
      await prisma.ussdSession.update({ where: { id: s.id }, data: { lastMenu: "MAIN", state: { step: "MAIN" } } });
      return Response.json({ ok: true, menu: `Sent P ${amount.toFixed(2)} to ${toEmail}.\n\n${mainMenu()}` });
    } catch (e: any) {
      return Response.json({ ok: true, menu: `Failed: ${e?.message ?? "ERROR"}\n\n0. Back` });
    }
  }

  // PAY_MERCHANT
  if (s.lastMenu === "PAY_MERCHANT") {
    if (input === "0") {
      await prisma.ussdSession.update({ where: { id: s.id }, data: { lastMenu: "MAIN", state: { step: "MAIN" } } });
      return Response.json({ ok: true, menu: mainMenu() });
    }
    await prisma.ussdSession.update({
      where: { id: s.id },
      data: { lastMenu: "PAY_AMOUNT", state: { step: "PAY_AMOUNT", merchantEmail: input } },
    });
    return Response.json({ ok: true, menu: `Merchant: ${input}\nEnter amount (BWP):` });
  }

  if (s.lastMenu === "PAY_AMOUNT") {
    if (input === "0") {
      await prisma.ussdSession.update({ where: { id: s.id }, data: { lastMenu: "MAIN", state: { step: "MAIN" } } });
      return Response.json({ ok: true, menu: mainMenu() });
    }
    if (!user) return Response.json({ ok: true, menu: "No account linked to this phone." });
    const amount = Number(input);
    if (!Number.isFinite(amount) || amount <= 0) return Response.json({ ok: true, menu: "Invalid amount. Enter amount (BWP):" });
    const amountMinor = BigInt(Math.round(amount * 100));
    const merchantEmail = String(state.merchantEmail ?? "");
    try {
      await transferLefaToLefa({ fromUserId: user.id, toEmail: merchantEmail, amountMinor, description: "USSD merchant pay" });
      await prisma.ussdSession.update({ where: { id: s.id }, data: { lastMenu: "MAIN", state: { step: "MAIN" } } });
      return Response.json({ ok: true, menu: `Paid P ${amount.toFixed(2)} to ${merchantEmail}.\n\n${mainMenu()}` });
    } catch (e: any) {
      return Response.json({ ok: true, menu: `Failed: ${e?.message ?? "ERROR"}\n\n0. Back` });
    }
  }

  // fallback
  await prisma.ussdSession.update({ where: { id: s.id }, data: { lastMenu: "MAIN", state: { step: "MAIN" } } });
  return Response.json({ ok: true, menu: mainMenu() });
}
