import { TopNav } from "@/components/nav";
import { Button, Card, CardBody, CardHeader, Container, Input } from "@/components/ui";
import { requireSession } from "@/lib/session";
import { prisma } from "@/lib/prisma";
import { randomToken } from "@/lib/crypto";
import { z } from "zod";

const Schema = z.object({
  toPhone: z.string().min(7).max(20),
  amount: z.coerce.number().positive(),
  reference: z.string().max(64).optional(),
  description: z.string().max(140).optional(),
  channel: z.enum(["WHATSAPP", "SMS"]).optional().default("WHATSAPP"),
});

export default async function TextPayPage() {
  const session = await requireSession();

  async function create(_: any, formData: FormData) {
    "use server";
    const s = await requireSession();
    const parsed = Schema.safeParse({
      toPhone: formData.get("toPhone"),
      amount: formData.get("amount"),
      reference: formData.get("reference"),
      description: formData.get("description"),
      channel: formData.get("channel"),
    });
    if (!parsed.success) return { ok: false, error: "INVALID_INPUT" };
    const amountMinor = BigInt(Math.round(parsed.data.amount * 100));
    const token = randomToken(18);
    const r = await prisma.textPayRequest.create({
      data: {
        fromUserId: s.uid,
        toPhone: parsed.data.toPhone.trim(),
        amount: amountMinor,
        currency: "BWP",
        reference: parsed.data.reference ?? null,
        description: parsed.data.description ?? null,
        channel: parsed.data.channel,
        status: "SENT",
        token,
      },
    });
    return { ok: true, url: `/t/pay/${r.token}` };
  }

  const sent = await prisma.textPayRequest.findMany({
    where: { fromUserId: session.uid },
    orderBy: { createdAt: "desc" },
    take: 10,
  });

  return (
    <>
      <TopNav />
      <Container>
        <div className="mx-auto max-w-2xl">
          <Card>
            <CardHeader title="Text-to-Pay" subtitle="Request money by link (WhatsApp/SMS)" />
            <CardBody>
              <form action={create} className="grid gap-3">
                <Input name="toPhone" label="Recipient phone" placeholder="+2677xxxxxxx" />
                <Input name="amount" label="Amount (BWP)" placeholder="200" type="number" />
                <Input name="reference" label="Reference (optional)" placeholder="e.g., Taxi / Order 1001" />
                <Input name="description" label="Description (optional)" placeholder="Short note" />
                <div>
                  <div className="mb-1 text-xs font-medium text-zinc-700">Channel</div>
                  <select name="channel" defaultValue="WHATSAPP" className="w-full rounded-xl border border-zinc-200 bg-white px-3 py-2 text-sm">
                    <option value="WHATSAPP">WhatsApp</option>
                    <option value="SMS">SMS</option>
                  </select>
                </div>
                <Button type="submit">Create request</Button>
              </form>

              <div className="mt-6">
                <div className="text-sm font-medium">Recent requests</div>
                <div className="mt-2 grid gap-2">
                  {sent.map((r) => (
                    <div key={r.id} className="rounded-xl border border-zinc-200 p-3">
                      <div className="text-sm font-medium">To {r.toPhone} • P {Number(r.amount) / 100}</div>
                      <div className="text-xs text-zinc-600">Status: {r.status} • {new Date(r.createdAt).toLocaleString()}</div>
                      <div className="mt-2 flex flex-wrap gap-2">
                        <a className="text-sm underline" href={`/t/pay/${r.token}`}>Open link</a>
                        <a
                          className="text-sm underline"
                          href={`https://wa.me/?text=${encodeURIComponent(`LEFA payment request: ${process.env.NEXT_PUBLIC_BASE_URL ?? ""}/t/pay/${r.token}`)}`}
                          target="_blank"
                          rel="noreferrer"
                        >
                          Share WhatsApp
                        </a>
                      </div>
                    </div>
                  ))}
                </div>
              </div>
            </CardBody>
          </Card>
        </div>
      </Container>
    </>
  );
}
