import Link from "next/link";
import { TopNav } from "@/components/nav";
import { Card, CardBody, CardHeader, Container } from "@/components/ui";
import { requireSession } from "@/lib/session";
import { requireMerchant } from "@/lib/role";
import { prisma } from "@/lib/prisma";
import { fromMinor } from "@/lib/money";

export default async function MerchantHome() {
  const session = await requireSession();
  const user = await requireMerchant(session.uid);

  const intents = await prisma.paymentIntent.findMany({
    where: { merchantId: user.merchant!.id },
    orderBy: { createdAt: "desc" },
    take: 10,
  });

  const wallet = await prisma.walletAccount.findUnique({ where: { userId: session.uid } });
  const balance = wallet?.balance ?? 0n;

  return (
    <>
      <TopNav />
      <Container>
        <div className="grid gap-6 md:grid-cols-2">
          <Card>
            <CardHeader title="Merchant" subtitle={user.merchant!.tradingName} />
            <CardBody>
              <div className="text-sm text-zinc-600">Settlement balance</div>
              <div className="mt-1 text-3xl font-semibold">P {fromMinor(balance)}</div>

              <div className="mt-4 flex gap-2">
                <Link
                  href="/merchant/api-keys"
                  className="rounded-xl bg-zinc-900 px-4 py-2 text-sm font-medium text-white"
                >
                  API Keys
                </Link>
                <Link
                  href="/merchant/payments"
                  className="rounded-xl border border-zinc-200 bg-white px-4 py-2 text-sm font-medium text-zinc-900"
                >
                  Payments
                </Link>
                <Link
                  href="/merchant/webhooks"
                  className="rounded-xl border border-zinc-200 bg-white px-4 py-2 text-sm font-medium text-zinc-900"
                >
                  Webhooks
                </Link>
              </div>
            </CardBody>
          </Card>

          <Card>
            <CardHeader title="Latest payment intents" subtitle="Last 10" />
            <CardBody>
              <div className="space-y-3">
                {intents.length === 0 ? (
                  <div className="text-sm text-zinc-600">No payment intents yet.</div>
                ) : (
                  intents.map((pi) => (
                    <div key={pi.id} className="rounded-xl border border-zinc-200 px-3 py-2">
                      <div className="flex items-center justify-between">
                        <div className="text-sm font-medium">{pi.reference}</div>
                        <div className="text-sm font-semibold">P {fromMinor(pi.amount)}</div>
                      </div>
                      <div className="mt-1 text-xs text-zinc-600">{pi.status} · {new Date(pi.createdAt).toLocaleString()}</div>
                    </div>
                  ))
                )}
              </div>
            </CardBody>
          </Card>
        </div>
      </Container>
    </>
  );
}
