import Link from "next/link";
import { TopNav } from "@/components/nav";
import { Button, Card, CardBody, CardHeader, Container, Input, Label } from "@/components/ui";
import { requireSession } from "@/lib/session";
import { prisma } from "@/lib/prisma";
import { createInStoreChargeAction, createLocationAction, openShiftAction, closeShiftAction } from "@/app/actions/merchant";
import { fromMinor } from "@/lib/money";

export default async function MerchantPosPage() {
  const session = await requireSession();

  const merchant = await prisma.merchantProfile.findUnique({ where: { userId: session.uid } });
  if (!merchant) {
    return (
      <>
        <TopNav />
        <Container>
          <Card>
            <CardHeader title="POS" subtitle="Merchant required" />
            <CardBody>
              <div className="text-sm text-zinc-600">Create a merchant profile first.</div>
              <div className="mt-3">
                <Link className="text-sm underline" href="/merchant">
                  Go to Merchant dashboard
                </Link>
              </div>
            </CardBody>
          </Card>
        </Container>
      </>
    );
  }

  const locations = await prisma.merchantLocation.findMany({
    where: { merchantId: merchant.id },
    orderBy: { createdAt: "desc" },
  });

  const openShift = locations.length
    ? await prisma.cashierShift.findFirst({
        where: { merchantId: merchant.id, cashierUserId: session.uid, closedAt: null },
        orderBy: { openedAt: "desc" },
      })
    : null;

  const recentCharges = await prisma.inStoreCharge.findMany({
    where: { merchantId: merchant.id },
    orderBy: { createdAt: "desc" },
    take: 10,
  });

  return (
    <>
      <TopNav />
      <Container>
        <div className="grid gap-6 lg:grid-cols-2">
          <Card>
            <CardHeader title="In-store POS" subtitle="Create a charge, show QR, get paid" />
            <CardBody>
              {locations.length === 0 ? (
                <div className="space-y-3">
                  <div className="text-sm text-zinc-600">Create your first location to start POS.</div>
                  <form action={createLocationAction} className="grid gap-3">
                    <div>
                      <Label>Location name</Label>
                      <Input name="name" placeholder="Main Stall" />
                    </div>
                    <div className="grid grid-cols-2 gap-3">
                      <div>
                        <Label>City</Label>
                        <Input name="city" placeholder="Gaborone" />
                      </div>
                      <div>
                        <Label>Address</Label>
                        <Input name="address" placeholder="Westgate" />
                      </div>
                    </div>
                    <Button type="submit">Create location</Button>
                  </form>
                </div>
              ) : (
                <div className="space-y-5">
                  <div className="rounded-xl border border-zinc-200 bg-zinc-50 p-4">
                    <div className="text-sm font-medium">Cashier shift</div>
                    <div className="mt-1 text-xs text-zinc-600">Open a shift for clean reconciliation.</div>
                    {openShift ? (
                      <div className="mt-2 flex items-center justify-between">
                        <div className="text-sm">
                          Open: <span className="font-medium">{new Date(openShift.openedAt).toLocaleString()}</span>
                        </div>
                        <form action={closeShiftAction}>
                          <input type="hidden" name="shiftId" value={openShift.id} />
                          <Button type="submit" variant="secondary">
                            Close shift
                          </Button>
                        </form>
                      </div>
                    ) : (
                      <form action={openShiftAction} className="mt-3 grid gap-3">
                        <div>
                          <Label>Location</Label>
                          <select
                            name="locationId"
                            className="mt-1 w-full rounded-xl border border-zinc-200 bg-white px-3 py-2 text-sm"
                            defaultValue={locations[0]?.id}
                          >
                            {locations.map((l) => (
                              <option key={l.id} value={l.id}>
                                {l.name}
                              </option>
                            ))}
                          </select>
                        </div>
                        <div>
                          <Label>Note (optional)</Label>
                          <Input name="note" placeholder="Morning shift" />
                        </div>
                        <Button type="submit">Open shift</Button>
                      </form>
                    )}
                  </div>

                  <div className="rounded-xl border border-zinc-200 bg-zinc-50 p-4">
                    <div className="text-sm font-medium">Create a charge</div>
                    <div className="mt-1 text-xs text-zinc-600">
                      Enter amount → generate QR → customer scans and pays.
                    </div>
                    <form action={createInStoreChargeAction} className="mt-3 grid gap-3">
                      <div>
                        <Label>Location</Label>
                        <select
                          name="locationId"
                          className="mt-1 w-full rounded-xl border border-zinc-200 bg-white px-3 py-2 text-sm"
                          defaultValue={locations[0]?.id}
                        >
                          {locations.map((l) => (
                            <option key={l.id} value={l.id}>
                              {l.name}
                            </option>
                          ))}
                        </select>
                      </div>
                      <div className="grid grid-cols-2 gap-3">
                        <div>
                          <Label>Amount (BWP)</Label>
                          <Input name="amount" type="number" placeholder="150" />
                        </div>
                        <div>
                          <Label>Reference (optional)</Label>
                          <Input name="reference" placeholder="SALE-102" />
                        </div>
                      </div>
                      <div>
                        <Label>Description (optional)</Label>
                        <Input name="description" placeholder="2x Burger" />
                      </div>
                      <Button type="submit">Generate QR</Button>
                      <div className="text-xs text-zinc-500">
                        After submit, you’ll be redirected to the QR screen.
                      </div>
                    </form>
                  </div>
                </div>
              )}
            </CardBody>
          </Card>

          <Card>
            <CardHeader title="Recent POS charges" subtitle="Last 10" />
            <CardBody>
              {recentCharges.length === 0 ? (
                <div className="text-sm text-zinc-600">No POS charges yet.</div>
              ) : (
                <div className="space-y-3">
                  {recentCharges.map((c) => (
                    <div key={c.id} className="flex items-center justify-between rounded-xl border border-zinc-200 px-3 py-2">
                      <div>
                        <div className="text-sm font-medium">P {fromMinor(c.amount)}</div>
                        <div className="text-xs text-zinc-600">
                          {c.status} · {new Date(c.createdAt).toLocaleString()}
                        </div>
                      </div>
                      <Link className="text-sm underline" href={`/merchant/pos/qr/${c.token}`}>
                        View
                      </Link>
                    </div>
                  ))}
                </div>
              )}
            </CardBody>
          </Card>
        </div>
      </Container>
    </>
  );
}
