import { TopNav } from "@/components/nav";
import { Card, CardBody, CardHeader, Container } from "@/components/ui";
import { requireSession } from "@/lib/session";
import { requireMerchant } from "@/lib/role";
import { prisma } from "@/lib/prisma";
import { CreateWebhookForm, DisableWebhookButton } from "./WebhooksClient";

export default async function WebhooksPage() {
  const session = await requireSession();
  await requireMerchant(session.uid);

  const hooks = await prisma.webhookEndpoint.findMany({
    where: { userId: session.uid },
    orderBy: { createdAt: "desc" },
  });

  return (
    <>
      <TopNav />
      <Container>
        <div className="grid gap-6 md:grid-cols-2">
          <Card>
            <CardHeader
              title="Webhooks"
              subtitle="Receive payment confirmations on your server"
            />
            <CardBody>
              <div className="text-sm text-zinc-700">
                Events: <span className="font-medium">payment_intent.succeeded</span>, <span className="font-medium">payment_request.paid</span>
              </div>
              <div className="mt-2 text-xs text-zinc-600">
                LEFA signs requests with <span className="font-mono">X-Lefa-Signature</span> (HMAC-SHA256 of the JSON body using your webhook secret).
              </div>
              <div className="mt-4">
                <CreateWebhookForm />
              </div>
            </CardBody>
          </Card>

          <Card>
            <CardHeader title="Endpoints" subtitle="Active and disabled" />
            <CardBody>
              <div className="space-y-3">
                {hooks.length === 0 ? (
                  <div className="text-sm text-zinc-600">No webhook endpoints yet.</div>
                ) : (
                  hooks.map((h) => (
                    <div key={h.id} className="rounded-xl border border-zinc-200 p-3">
                      <div className="text-sm font-medium break-all">{h.url}</div>
                      <div className="mt-1 text-xs text-zinc-600">Events: {h.events || "(all)"}</div>
                      <div className="mt-1 text-xs text-zinc-600">Created: {new Date(h.createdAt).toLocaleString()}</div>
                      {h.disabledAt ? (
                        <div className="mt-2 text-xs text-red-600">Disabled</div>
                      ) : (
                        <div className="mt-2 flex justify-end">
                          <DisableWebhookButton id={h.id} />
                        </div>
                      )}
                    </div>
                  ))
                )}
              </div>
            </CardBody>
          </Card>
        </div>
      </Container>
    </>
  );
}
