import type { NextAuthOptions } from "next-auth";
import CredentialsProvider from "next-auth/providers/credentials";
import { prisma } from "@/lib/prisma";
import { verifyPassword } from "@/lib/crypto";

export const authOptions: NextAuthOptions = {
  session: { strategy: "jwt" },
  providers: [
    CredentialsProvider({
      name: "Credentials",
      credentials: {
        loginType: { label: "Login Type", type: "text" },
        email: { label: "Email", type: "email" },
        password: { label: "Password", type: "password" },
        phone: { label: "Phone", type: "text" },
        otp: { label: "OTP", type: "text" },
        challengeId: { label: "Challenge ID", type: "text" },
      },
      async authorize(credentials) {
        const loginType = (credentials?.loginType ?? "email").toString();

        if (loginType === "phone") {
          const phone = (credentials?.phone ?? "").toString().trim();
          const otp = (credentials?.otp ?? "").toString().trim();
          const challengeId = (credentials?.challengeId ?? "").toString().trim();
          if (!phone || !otp || !challengeId) return null;

          const ch = await prisma.otpChallenge.findUnique({ where: { id: challengeId } });
          if (!ch || ch.purpose !== "LOGIN") return null;
          if (ch.phone !== phone) return null;
          if (ch.verifiedAt) return null;
          if (ch.expiresAt.getTime() < Date.now()) return null;
          if (ch.code !== otp) return null;

          await prisma.otpChallenge.update({ where: { id: ch.id }, data: { verifiedAt: new Date() } });

          const existing = await prisma.user.findUnique({ where: { phone } });
          const user =
            existing ??
            (await prisma.user.create({
              data: {
                phone,
                passwordHash: null,
                fullName: null,
                role: "CONSUMER",
                kycLevel: "BASIC",
                kycProfile: { create: { status: "APPROVED", phoneVerifiedAt: new Date() } },
                wallet: { create: { balance: BigInt(0), currency: "BWP" } },
              },
            }));

          await prisma.kycProfile.upsert({
            where: { userId: user.id },
            update: { phoneVerifiedAt: new Date() },
            create: { userId: user.id, status: "APPROVED", phoneVerifiedAt: new Date() },
          });
          if (user.kycLevel === "UNVERIFIED") {
            await prisma.user.update({ where: { id: user.id }, data: { kycLevel: "BASIC" } });
          }

          return {
            id: user.id,
            email: user.email ?? undefined,
            name: user.fullName ?? user.phone ?? "LEFA User",
            role: user.role,
          } as any;
        }

        // Email/password login (legacy + admin)
        const email = credentials?.email?.toLowerCase().trim();
        const password = credentials?.password ?? "";
        if (!email || !password) return null;

        const user = await prisma.user.findUnique({ where: { email } });
        if (!user || !user.passwordHash) return null;

        const ok = await verifyPassword(password, user.passwordHash);
        if (!ok) return null;

        return {
          id: user.id,
          email: user.email ?? undefined,
          name: user.fullName ?? user.email ?? user.phone ?? "LEFA User",
          role: user.role,
        } as any;
      },
    }),
  ],
  callbacks: {
    async jwt({ token, user }) {
      if (user) {
        token.uid = (user as any).id;
        token.role = (user as any).role;
      }
      return token;
    },
    async session({ session, token }) {
      (session as any).uid = token.uid;
      (session as any).role = token.role;
      return session;
    },
  },
  pages: {
    signIn: "/signin",
  },
};
