import { prisma } from "@/lib/prisma";
import { dispatchUserWebhooks } from "@/lib/webhooks";

export async function payPaymentIntent(params: {
  payerUserId: string;
  checkoutToken: string;
}) {
  const { payerUserId, checkoutToken } = params;

  const result = await prisma.$transaction(async (tx) => {
    const pi = await tx.paymentIntent.findUnique({
      where: { checkoutToken },
      include: { merchant: { include: { user: true } } },
    });

    if (!pi) throw new Error("PAYMENT_INTENT_NOT_FOUND");
    if (pi.status !== "CREATED" && pi.status !== "REQUIRES_PAYMENT") {
      throw new Error("PAYMENT_INTENT_NOT_PAYABLE");
    }

    const payerWallet = await tx.walletAccount.upsert({
      where: { userId: payerUserId },
      update: {},
      create: { userId: payerUserId, balance: 0n, currency: "BWP" },
    });

    const merchantUserId = pi.merchant.userId;
    const merchantWallet = await tx.walletAccount.upsert({
      where: { userId: merchantUserId },
      update: {},
      create: { userId: merchantUserId, balance: 0n, currency: "BWP" },
    });

    if (payerWallet.balance < pi.amount) throw new Error("INSUFFICIENT_FUNDS");

    await tx.walletAccount.update({
      where: { id: payerWallet.id },
      data: { balance: { decrement: pi.amount } },
    });

    await tx.walletAccount.update({
      where: { id: merchantWallet.id },
      data: { balance: { increment: pi.amount } },
    });

    const t = await tx.transaction.create({
      data: {
        type: "PAYMENT",
        status: "POSTED",
        amount: pi.amount,
        currency: pi.currency,
        description: pi.description ?? `Payment ${pi.reference}`,
        fromUserId: payerUserId,
        toUserId: merchantUserId,
        reference: `pi_${pi.id}`,
      },
    });

    await tx.paymentIntent.update({
      where: { id: pi.id },
      data: { status: "SUCCEEDED" },
    });

    return {
      paymentIntentId: pi.id,
      txId: t.id,
      merchantUserId,
      checkoutToken: pi.checkoutToken,
      amount: pi.amount,
      currency: pi.currency,
      reference: pi.reference,
    };
  });

  await dispatchUserWebhooks(result.merchantUserId, "payment_intent.succeeded", {
    checkoutToken: result.checkoutToken,
    amount: result.amount.toString(),
    currency: result.currency,
    reference: result.reference,
    txId: result.txId,
    payerUserId,
  });

  return { paymentIntentId: result.paymentIntentId, txId: result.txId };
}
