import { prisma } from "@/lib/prisma";
import { limitsFor, isOutgoingType } from "@/lib/limits";

function startOfToday() {
  const d = new Date();
  d.setHours(0, 0, 0, 0);
  return d;
}

export async function logRiskEvent(params: {
  userId?: string | null;
  type: "LIMIT_EXCEEDED" | "VELOCITY_ALERT" | "NEW_DEVICE" | "SUSPICIOUS_PATTERN";
  severity?: number;
  message: string;
  meta?: any;
}) {
  const { userId, type, severity, message, meta } = params;
  await prisma.riskEvent.create({
    data: {
      userId: userId ?? null,
      type,
      severity: severity ?? 1,
      message,
      meta: meta ?? undefined,
    },
  });
}

export async function assertSpendingAllowed(params: {
  userId: string;
  amountMinor: bigint;
  txType: "TRANSFER" | "PAYMENT" | "WITHDRAWAL";
}) {
  const { userId, amountMinor, txType } = params;
  if (amountMinor <= 0n) throw new Error("INVALID_AMOUNT");

  const user = await prisma.user.findUnique({ where: { id: userId }, select: { kycLevel: true } });
  if (!user) throw new Error("USER_NOT_FOUND");

  const limits = limitsFor(user.kycLevel);
  if (amountMinor > limits.perTxMinor) {
    await logRiskEvent({
      userId,
      type: "LIMIT_EXCEEDED",
      severity: 2,
      message: `Per-transaction limit exceeded for ${txType}`,
      meta: { amountMinor: amountMinor.toString(), perTxMinor: limits.perTxMinor.toString(), kycLevel: user.kycLevel },
    });
    throw new Error("LIMIT_PER_TX_EXCEEDED");
  }

  const today = startOfToday();
  const outgoing = await prisma.transaction.findMany({
    where: {
      fromUserId: userId,
      status: "POSTED",
      createdAt: { gte: today },
    },
    select: { amount: true, type: true },
  });

  let spent = 0n;
  for (const t of outgoing) {
    if (isOutgoingType(t.type)) spent += t.amount;
  }

  if (spent + amountMinor > limits.dailyOutMinor) {
    await logRiskEvent({
      userId,
      type: "VELOCITY_ALERT",
      severity: 2,
      message: `Daily outgoing limit exceeded for ${txType}`,
      meta: {
        amountMinor: amountMinor.toString(),
        spentTodayMinor: spent.toString(),
        dailyOutMinor: limits.dailyOutMinor.toString(),
        kycLevel: user.kycLevel,
      },
    });
    throw new Error("LIMIT_DAILY_EXCEEDED");
  }
}
